# 機能設計書 16-HTTP Digest認証

## 概要

本ドキュメントはFastAPIフレームワークにおけるHTTP Digest認証機能の設計仕様を定義する。

### 本機能の処理概要

HTTP Digest認証機能は、RFC 7616で定義されたDigest認証スキームをサポートする。HTTPDigestクラスをDependsまたはSecurity依存関係として使用することで、エンドポイントにDigest認証を適用できる。Digest認証はBasic認証と異なり、パスワードをネットワーク上で平文（Base64）で送信しないため、より安全な認証方式である。

**業務上の目的・背景**：Digest認証は、HTTP Basic認証のセキュリティ上の問題（パスワードが平文で送信される）を解決するために設計された認証方式である。チャレンジ-レスポンス方式を採用し、パスワードのMD5ハッシュを使用することで、ネットワーク上でのパスワード漏洩を防ぐ。ただし、現代ではHTTPS + Bearer認証が主流であり、Digest認証の使用は限定的である。

**機能の利用シーン**：
- レガシーシステムとの互換性が必要な場合
- HTTPSが使用できない環境での認証（非推奨）
- 特定のハードウェアデバイスとの連携
- OpenAPIドキュメントでのDigest認証スキーム定義

**主要な処理内容**：
1. HTTPリクエストからAuthorizationヘッダーを取得
2. Digestスキームの検証
3. クレデンシャル部分を抽出
4. HTTPAuthorizationCredentialsオブジェクトとして返却
5. 認証失敗時のHTTPException発生（auto_error=Trueの場合）

**関連システム・外部連携**：
- OpenAPI: HTTPセキュリティスキームの定義

**権限による制御**：Digest認証自体は権限制御を行わない。実際のチャレンジ-レスポンス検証は開発者が実装する必要がある。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | Swagger UI | 補助機能 | HTTP Digest認証のスキーム表示（実際の認証フローはサポートされない場合がある） |

## 機能種別

セキュリティ / 認証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| scheme_name | str \| None | No | セキュリティスキーム名 | 文字列 |
| description | str \| None | No | セキュリティスキームの説明 | 文字列 |
| auto_error | bool | No | 認証失敗時に自動でエラーを発生させるか（デフォルト: True） | ブール値 |

### 入力データソース

- HTTPリクエストのAuthorizationヘッダー（`Digest <credentials>`）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| HTTPAuthorizationCredentials | Pydantic BaseModel | スキームとクレデンシャルを含むオブジェクト |
| credentials.scheme | str | 認証スキーム（"Digest"） |
| credentials.credentials | str | Digestクレデンシャル文字列 |

### 出力先

- パスオペレーション関数のパラメータ
- OpenAPIスキーマ（セキュリティスキーム定義）

## 処理フロー

### 処理シーケンス

```
1. リクエスト受信
   └─ Authorization: Digest <credentials> ヘッダーの確認
2. ヘッダー解析
   └─ get_authorization_scheme_param関数でスキームとクレデンシャルを分離
3. スキーム検証
   └─ "digest"（大文字小文字区別なし）であることを確認
4. 結果返却
   └─ HTTPAuthorizationCredentialsオブジェクトを生成して返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト受信] --> B{Authorizationヘッダーあり?}
    B -->|No| C{auto_error?}
    B -->|Yes| D[スキームとクレデンシャルを分離]
    D --> E{スキームはdigest?}
    E -->|No| C
    E -->|Yes| F{クレデンシャルが空?}
    F -->|Yes| C
    F -->|No| G[HTTPAuthorizationCredentials生成]
    G --> H[返却]
    C -->|True| I[HTTPException 401発生]
    C -->|False| J[None返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-16-01 | Digestスキーム検証 | Authorizationヘッダーのスキームは"digest"（大文字小文字区別なし）である必要がある | HTTPDigest使用時 |
| BR-16-02 | クレデンシャル必須 | クレデンシャル部分は空であってはならない | 認証情報の検証時 |
| BR-16-03 | WWW-Authenticateヘッダー | 認証エラー時はWWW-Authenticate: Digestヘッダーを含める | HTTPException発生時 |

### 計算ロジック

特になし。FastAPIのHTTPDigestクラスはAuthorizationヘッダーの抽出のみを行い、実際のDigest認証のチャレンジ-レスポンス検証は開発者が実装する必要がある。

## データベース操作仕様

本機能はデータベース操作を直接行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| HTTP 401 | Unauthorized | Authorizationヘッダーがない | 正しいクレデンシャルを送信 |
| HTTP 401 | Unauthorized | スキームがdigestでない | Digest形式で送信 |
| HTTP 401 | Unauthorized | クレデンシャル部分が空 | 有効なクレデンシャルを送信 |
| HTTP 403 | Forbidden | 無効なクレデンシャル（スキームが異なる場合） | 正しいスキームで送信 |

### リトライ仕様

本機能では自動リトライは行わない。

## トランザクション仕様

本機能はトランザクション管理を行わない。

## パフォーマンス要件

- 文字列分離のみの軽量処理
- 実際のDigest検証を実装する場合はMD5ハッシュ計算が必要

## セキュリティ考慮事項

- FastAPIのHTTPDigestクラスはAuthorizationヘッダーの抽出のみを行い、チャレンジ-レスポンス検証は開発者が実装する必要がある
- Digest認証はMD5を使用しており、現代の基準では十分に安全とは言えない
- 可能であればHTTPS + Bearer認証（JWT等）の使用を推奨
- Digest認証を使用する場合でも、HTTPSと併用することを強く推奨

## 備考

- FastAPIのHTTPDigestクラスはOpenAPI互換のスキーム定義とAuthorizationヘッダーの抽出を提供するが、完全なDigest認証フローの実装は含まれていない
- 実際のDigest認証を実装する場合は、nonce、realm、opaque等のパラメータ管理とレスポンス検証が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、認証情報を表すデータ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | http.py | `fastapi/security/http.py` | HTTPAuthorizationCredentialsクラスの定義（29-66行目） |
| 1-2 | models.py | `fastapi/openapi/models.py` | HTTPBaseモデルの定義（348-351行目） |

**読解のコツ**: HTTPDigestはHTTPBearerと同様にHTTPAuthorizationCredentialsを返す。

#### Step 2: HTTPDigestクラスを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | http.py | `fastapi/security/http.py` | HTTPDigestクラスの定義（322-399行目） |

**主要処理フロー**:
1. **346-376行目**: __init__メソッドでHTTPBaseModel(scheme="digest")を設定
2. **378-384行目**: make_not_authenticated_errorメソッドでエラー生成
3. **386-399行目**: __call__メソッドでDigestクレデンシャルを抽出

#### Step 3: 認証フローを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | http.py | `fastapi/security/http.py` | __call__メソッド内の検証ロジック（386-399行目） |

**主要処理フロー**:
- **387行目**: Authorizationヘッダーの取得
- **388行目**: get_authorization_scheme_paramでスキームとクレデンシャルを分離
- **389-390行目**: スキームが"digest"でない場合の処理
- **391-398行目**: クレデンシャルが空の場合の処理
- **399行目**: HTTPAuthorizationCredentialsの返却

### プログラム呼び出し階層図

```
HTTPDigest(scheme_name, description, auto_error)
    │
    └─ HTTPBase.__init__()（暗黙的）
           │
           └─ __call__(request)
                  │
                  ├─ request.headers.get("Authorization")
                  │
                  ├─ get_authorization_scheme_param()
                  │      └─ スキームとクレデンシャル分離
                  │
                  └─ スキーム検証（"digest"）
                         │
                         └─ HTTPAuthorizationCredentials(scheme, credentials)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTTPリクエスト          HTTPDigest                HTTPAuthorizationCredentials
Authorization:    ───▶   __call__()          ───▶   {scheme: "Digest",
Digest username=...         │                        credentials: "username=..."}
                            ▼
                   get_authorization_scheme_param()
                   スキームとクレデンシャル分離
                            │
                            ▼
                   スキーム検証（"digest"）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| http.py | `fastapi/security/http.py` | ソース | HTTPDigestクラスとHTTPAuthorizationCredentialsの定義 |
| base.py | `fastapi/security/base.py` | ソース | SecurityBaseクラスの定義 |
| models.py | `fastapi/openapi/models.py` | ソース | HTTPBaseモデルの定義 |
| utils.py | `fastapi/security/utils.py` | ソース | get_authorization_scheme_param関数 |
| exceptions.py | `fastapi/exceptions.py` | ソース | HTTPException定義 |
